<?php
/* --------------------------------------------------------------
   TrackingCodeWriter.php 2020-03-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\TrackingCode\Repository;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\ConnectionException;
use Exception;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCodeId;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCodeIds;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCode;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCodes;
use RuntimeException;

/**
 * Class TrackingCodeWriter
 *
 * @package Gambio\Admin\TrackingCode\Repository
 */
class TrackingCodeWriter
{
    /**
     * @var Connection
     */
    private $db;
    
    
    /**
     * TrackingCodeWriter constructor.
     *
     * @param Connection $db
     */
    public function __construct(Connection $db)
    {
        $this->db = $db;
    }
    
    
    /**
     * @param TrackingCode $trackingCode
     *
     * @return int
     */
    public function insert(TrackingCode $trackingCode): int
    {
        $this->db->createQueryBuilder()
            ->insert('orders_parcel_tracking_codes')
            ->setValue('orders_parcel_tracking_code_id', ':id')
            ->setValue('order_id', ':orderId')
            ->setValue('tracking_code', ':code')
            ->setValue('language_id', ':languageId')
            ->setValue('parcel_service_id', ':parcelServiceId')
            ->setValue('parcel_service_name', ':parcelServiceName')
            ->setValue('url', ':parcelServiceUrl')
            ->setValue('comment', ':parcelServiceComment')
            ->setValue('creation_date', ':createdOn')
            ->setParameter('id', $trackingCode->id())
            ->setParameter('orderId', $trackingCode->orderId())
            ->setParameter('code', $trackingCode->code())
            ->setParameter('languageId', $trackingCode->languageId())
            ->setParameter('parcelServiceId', $trackingCode->parcelServiceId())
            ->setParameter('parcelServiceName', $trackingCode->parcelServiceName())
            ->setParameter('parcelServiceUrl', $trackingCode->url())
            ->setParameter('parcelServiceComment', $trackingCode->comment())
            ->setParameter('createdOn', $trackingCode->createdOn())
            ->execute();
        $id = $this->db->createQueryBuilder()->getConnection()->lastInsertId();
        
        return (int)$id;
    }
    
    
    /**
     * @param TrackingCodes $trackingCodes
     *
     * @return int[]
     *
     * @throws ConnectionException
     */
    public function insertMultiple(TrackingCodes $trackingCodes): array
    {
        $ids = [];
        try {
            $this->db->beginTransaction();
            foreach ($trackingCodes as $reference => $trackingCode) {
                
                $ids[$reference] = $this->insert($trackingCode);
            }
            $this->db->commit();
        } catch (Exception $exception) {
            $this->db->rollBack();
            throw new RuntimeException('Could not add the parcel service', $exception->getCode(), $exception);
        }
        
        return $ids;
    }
    
    
    /**
     * @param TrackingCode $trackingCode
     */
    public function update(TrackingCode $trackingCode): void
    {
        $this->db->createQueryBuilder()
            ->update('orders_parcel_tracking_codes')
            ->set('order_id', ':orderId')
            ->set('tracking_code', ':code')
            ->set('language_id', ':languageId')
            ->set('parcel_service_id', ':parcelServiceId')
            ->set('parcel_service_name', ':parcelServiceName')
            ->set('url', ':parcelServiceUrl')
            ->set('comment', ':parcelServiceComment')
            ->where('orders_parcel_tracking_code_id = :id')
            ->setParameter('id', $trackingCode->id())
            ->setParameter('orderId', $trackingCode->orderId())
            ->setParameter('code', $trackingCode->code())
            ->setParameter('languageId', $trackingCode->languageId())
            ->setParameter('parcelServiceId', $trackingCode->parcelServiceId())
            ->setParameter('parcelServiceName', $trackingCode->parcelServiceName())
            ->setParameter('parcelServiceUrl', $trackingCode->url())
            ->setParameter('parcelServiceComment', $trackingCode->comment())
            ->execute();
    }
    
    
    /**
     * @param TrackingCodes $trackingCodes
     *
     * @throws ConnectionException
     */
    public function updateMultiple(TrackingCodes $trackingCodes): void
    {
        try {
            $this->db->beginTransaction();
            foreach ($trackingCodes as $trackingCode) {
                $this->update($trackingCode);
            }
            $this->db->commit();
        } catch (Exception $exception) {
            $this->db->rollBack();
            throw new RuntimeException('Could not update the parcel service', $exception->getCode(), $exception);
        }
    }
    
    
    /**
     * @param TrackingCodeId $id
     */
    public function delete(TrackingCodeId $id): void
    {
        $this->db->createQueryBuilder()
            ->delete('orders_parcel_tracking_codes')
            ->where('orders_parcel_tracking_code_id = :id')
            ->setParameter('id', $id->value())
            ->execute();
    }
    
    
    /**
     * @param TrackingCodeIds $ids
     *
     * @throws ConnectionException
     */
    public function deleteMultiple(TrackingCodeIds $ids): void
    {
        try {
            $this->db->beginTransaction();
            foreach ($ids as $id) {
                $this->delete($id);
            }
            $this->db->commit();
        } catch (Exception $exception) {
            $this->db->rollBack();
            throw new RuntimeException('Could not delete the parcel service', $exception->getCode(), $exception);
        }
    }
}